/*
 * Decompiled with CFR 0.152.
 */
package de.sarocesch.plushie.util;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.mojang.logging.LogUtils;
import java.io.BufferedReader;
import java.io.InputStreamReader;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Base64;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import org.slf4j.Logger;

public class SkinFetcher {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static final String UUID_API = "https://api.mojang.com/users/profiles/minecraft/";
    private static final String PROFILE_API = "https://sessionserver.mojang.com/session/minecraft/profile/";
    private static final String CRAFATAR_SKIN_API = "https://crafatar.com/skins/";
    private static final Map<String, String> SKIN_CACHE = new ConcurrentHashMap<String, String>();
    private static final Map<String, Boolean> SLIM_CACHE = new ConcurrentHashMap<String, Boolean>();
    private static final Map<String, String> UUID_CACHE = new ConcurrentHashMap<String, String>();
    private static final Map<String, Long> CACHE_TIMES = new ConcurrentHashMap<String, Long>();
    private static final long CACHE_DURATION = 600000L;
    private static final int MAX_RETRIES = 3;
    private static final int RETRY_DELAY_MS = 1000;
    private static final int CONNECTION_TIMEOUT = 10000;
    private static final int READ_TIMEOUT = 10000;

    public static void fetchSkinUrl(String playerName, Consumer<String> callback) {
        SkinFetcher.fetchSkinData(playerName, (url, isSlim) -> callback.accept((String)url));
    }

    public static void fetchSkinData(String playerName, BiConsumer<String, Boolean> callback) {
        Long cacheTime;
        if (playerName == null || playerName.isEmpty()) {
            callback.accept("", false);
            return;
        }
        String lowerName = playerName.toLowerCase();
        if (SKIN_CACHE.containsKey(lowerName) && (cacheTime = CACHE_TIMES.get(lowerName)) != null && System.currentTimeMillis() - cacheTime < 600000L) {
            LOGGER.debug("Using cached skin data for player: {}", (Object)playerName);
            callback.accept(SKIN_CACHE.get(lowerName), SLIM_CACHE.getOrDefault(lowerName, false));
            return;
        }
        CompletableFuture.runAsync(() -> {
            String skinUrl = "";
            boolean isSlim = false;
            for (int attempt = 1; attempt <= 3; ++attempt) {
                try {
                    SkinData data = SkinFetcher.fetchSkinDataFromMojang(playerName);
                    if (data == null || data.url.isEmpty()) continue;
                    skinUrl = data.url;
                    isSlim = data.isSlim;
                    LOGGER.info("Fetched skin for {} (slim={}): {}", new Object[]{playerName, isSlim, skinUrl});
                }
                catch (Exception e) {
                    LOGGER.debug("Mojang API attempt {} failed for {}: {}", new Object[]{attempt, playerName, e.getMessage()});
                    if (attempt >= 3) continue;
                    try {
                        Thread.sleep(1000 * attempt);
                        continue;
                    }
                    catch (InterruptedException ie) {
                        Thread.currentThread().interrupt();
                    }
                }
                break;
            }
            if (skinUrl.isEmpty()) {
                LOGGER.debug("Mojang API failed, trying Crafatar fallback for: {}", (Object)playerName);
                try {
                    skinUrl = SkinFetcher.fetchSkinUrlFromCrafatar(playerName);
                    if (!skinUrl.isEmpty()) {
                        LOGGER.info("Fetched skin from Crafatar for {}: {}", (Object)playerName, (Object)skinUrl);
                        isSlim = false;
                    }
                }
                catch (Exception e) {
                    LOGGER.warn("Crafatar fallback also failed for {}: {}", (Object)playerName, (Object)e.getMessage());
                }
            }
            SKIN_CACHE.put(lowerName, skinUrl);
            SLIM_CACHE.put(lowerName, isSlim);
            CACHE_TIMES.put(lowerName, System.currentTimeMillis());
            callback.accept(skinUrl, isSlim);
        });
    }

    private static SkinData fetchSkinDataFromMojang(String playerName) throws Exception {
        String uuid = SkinFetcher.getUuidFromName(playerName);
        if (uuid == null || uuid.isEmpty()) {
            return null;
        }
        return SkinFetcher.getSkinDataFromUuid(uuid);
    }

    private static String fetchSkinUrlFromCrafatar(String playerName) throws Exception {
        String uuid = SkinFetcher.getUuidFromName(playerName);
        if (uuid == null || uuid.isEmpty()) {
            return "";
        }
        String crafatarUrl = CRAFATAR_SKIN_API + uuid;
        HttpURLConnection connection = (HttpURLConnection)new URL(crafatarUrl).openConnection();
        connection.setRequestMethod("HEAD");
        connection.setConnectTimeout(10000);
        connection.setReadTimeout(10000);
        connection.setRequestProperty("User-Agent", "SarosPlushieMod/1.0");
        int responseCode = connection.getResponseCode();
        if (responseCode == 200) {
            return crafatarUrl;
        }
        return "";
    }

    private static String getUuidFromName(String playerName) throws Exception {
        String lowerName = playerName.toLowerCase();
        if (UUID_CACHE.containsKey(lowerName)) {
            return UUID_CACHE.get(lowerName);
        }
        URL url = new URL(UUID_API + playerName);
        HttpURLConnection connection = (HttpURLConnection)url.openConnection();
        connection.setRequestMethod("GET");
        connection.setConnectTimeout(10000);
        connection.setReadTimeout(10000);
        connection.setRequestProperty("User-Agent", "SarosPlushieMod/1.0");
        int responseCode = connection.getResponseCode();
        if (responseCode != 200) {
            return null;
        }
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream()));){
            String line;
            StringBuilder response = new StringBuilder();
            while ((line = reader.readLine()) != null) {
                response.append(line);
            }
            JsonObject json = JsonParser.parseString((String)response.toString()).getAsJsonObject();
            String uuid = json.get("id").getAsString();
            UUID_CACHE.put(lowerName, uuid);
            String string = uuid;
            return string;
        }
    }

    private static SkinData getSkinDataFromUuid(String uuid) throws Exception {
        URL url = new URL(PROFILE_API + uuid);
        HttpURLConnection connection = (HttpURLConnection)url.openConnection();
        connection.setRequestMethod("GET");
        connection.setConnectTimeout(10000);
        connection.setReadTimeout(10000);
        connection.setRequestProperty("User-Agent", "SarosPlushieMod/1.0");
        int responseCode = connection.getResponseCode();
        if (responseCode != 200) {
            return null;
        }
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(connection.getInputStream()));){
            String line;
            StringBuilder response = new StringBuilder();
            while ((line = reader.readLine()) != null) {
                response.append(line);
            }
            JsonObject json = JsonParser.parseString((String)response.toString()).getAsJsonObject();
            JsonArray properties = json.getAsJsonArray("properties");
            for (JsonElement prop : properties) {
                JsonObject metadata;
                JsonObject propObj = prop.getAsJsonObject();
                if (!"textures".equals(propObj.get("name").getAsString())) continue;
                String encodedValue = propObj.get("value").getAsString();
                String decodedValue = new String(Base64.getDecoder().decode(encodedValue));
                JsonObject texturesJson = JsonParser.parseString((String)decodedValue).getAsJsonObject();
                JsonObject textures = texturesJson.getAsJsonObject("textures");
                if (!textures.has("SKIN")) continue;
                JsonObject skinObj = textures.getAsJsonObject("SKIN");
                String skinUrl = skinObj.get("url").getAsString();
                boolean isSlim = false;
                if (skinObj.has("metadata") && (metadata = skinObj.getAsJsonObject("metadata")).has("model")) {
                    isSlim = "slim".equals(metadata.get("model").getAsString());
                }
                SkinData skinData = new SkinData(skinUrl, isSlim);
                return skinData;
            }
        }
        return null;
    }

    public static boolean isSlimCached(String playerName) {
        if (playerName == null || playerName.isEmpty()) {
            return false;
        }
        return SLIM_CACHE.getOrDefault(playerName.toLowerCase(), false);
    }

    public static void clearCache() {
        SKIN_CACHE.clear();
        SLIM_CACHE.clear();
        UUID_CACHE.clear();
        CACHE_TIMES.clear();
    }

    private static class SkinData {
        final String url;
        final boolean isSlim;

        SkinData(String url, boolean isSlim) {
            this.url = url;
            this.isSlim = isSlim;
        }
    }
}

