/*
 * Decompiled with CFR 0.152.
 */
package com.crabmods.instantworldmirror.world;

import com.crabmods.instantworldmirror.InstantWorldMirror;
import com.crabmods.instantworldmirror.MirrorConfig;
import com.crabmods.instantworldmirror.entity.MirrorPortalEntity;
import com.crabmods.instantworldmirror.item.DimensionMirrorItem;
import com.crabmods.instantworldmirror.network.ClearMirrorEffectsPacket;
import com.crabmods.instantworldmirror.network.SyncMirrorEffectsPacket;
import com.crabmods.instantworldmirror.world.DimensionPool;
import com.crabmods.instantworldmirror.world.MirrorSession;
import com.crabmods.instantworldmirror.world.ModDimensions;
import com.crabmods.instantworldmirror.world.WorldCopyService;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import net.minecraft.core.BlockPos;
import net.minecraft.core.HolderLookup;
import net.minecraft.core.Vec3i;
import net.minecraft.core.particles.ParticleOptions;
import net.minecraft.core.particles.ParticleTypes;
import net.minecraft.core.registries.Registries;
import net.minecraft.nbt.CompoundTag;
import net.minecraft.nbt.ListTag;
import net.minecraft.nbt.Tag;
import net.minecraft.network.chat.Component;
import net.minecraft.network.protocol.common.custom.CustomPacketPayload;
import net.minecraft.resources.ResourceKey;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.sounds.SoundEvents;
import net.minecraft.sounds.SoundSource;
import net.minecraft.world.entity.Entity;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.block.Block;
import net.minecraft.world.level.block.Blocks;
import net.minecraft.world.level.block.state.BlockState;
import net.minecraft.world.level.material.Fluids;
import net.minecraft.world.phys.AABB;
import net.neoforged.neoforge.network.PacketDistributor;

public class MirrorWorldManager {
    private static final String SAVED_INVENTORY_KEY = "instantworldmirror_saved_inventory";
    private static final String SAVED_ENDERCHEST_KEY = "instantworldmirror_saved_enderchest";
    private static final String ORIGINAL_POS_KEY = "instantworldmirror_original_pos";
    private static final String ORIGINAL_DIM_KEY = "instantworldmirror_original_dim";
    private static final String SESSION_ID_KEY = "instantworldmirror_session_id";
    private static final ReadWriteLock sessionLock = new ReentrantReadWriteLock();
    private static final Map<UUID, MirrorSession> activeSessions = new ConcurrentHashMap<UUID, MirrorSession>();
    private static final Map<UUID, UUID> portalToSession = new ConcurrentHashMap<UUID, UUID>();
    private static final Map<UUID, UUID> playerToSession = new ConcurrentHashMap<UUID, UUID>();
    private static final Map<UUID, UUID> playerOwnedSession = new ConcurrentHashMap<UUID, UUID>();
    private static final Map<UUID, BlockPos> playerOriginalPositions = new ConcurrentHashMap<UUID, BlockPos>();
    private static final Map<UUID, ResourceKey<Level>> playerOriginalDimensions = new ConcurrentHashMap<UUID, ResourceKey<Level>>();
    private static final Map<UUID, Boolean> playerItemTransferPermission = new ConcurrentHashMap<UUID, Boolean>();
    private static final Map<UUID, Boolean> playerAccessDenied = new ConcurrentHashMap<UUID, Boolean>();
    private static final Set<UUID> playersBeingTeleported = ConcurrentHashMap.newKeySet();
    private static volatile boolean purgeMode = false;

    public static void enablePurgeMode() {
        purgeMode = true;
        InstantWorldMirror.LOGGER.info("Purge mode enabled - mirror world creation is now disabled");
    }

    public static void disablePurgeMode() {
        purgeMode = false;
        InstantWorldMirror.LOGGER.info("Purge mode disabled - mirror world creation is now enabled");
    }

    public static boolean isPurgeModeActive() {
        return purgeMode;
    }

    public static boolean hasActiveSession(UUID playerId) {
        return playerOwnedSession.containsKey(playerId) || playerToSession.containsKey(playerId);
    }

    public static Optional<MirrorSession> getPlayerOwnedSession(UUID playerId) {
        UUID sessionId = playerOwnedSession.get(playerId);
        if (sessionId != null) {
            return Optional.ofNullable(activeSessions.get(sessionId));
        }
        return Optional.empty();
    }

    public static Optional<MirrorSession> getPlayerCurrentSession(UUID playerId) {
        UUID sessionId = playerToSession.get(playerId);
        if (sessionId != null) {
            return Optional.ofNullable(activeSessions.get(sessionId));
        }
        return Optional.empty();
    }

    public static Optional<MirrorSession> getSession(UUID sessionId) {
        return Optional.ofNullable(activeSessions.get(sessionId));
    }

    public static Optional<MirrorSession> getSessionByPortal(UUID portalEntityId) {
        UUID sessionId = portalToSession.get(portalEntityId);
        if (sessionId != null) {
            return Optional.ofNullable(activeSessions.get(sessionId));
        }
        return Optional.empty();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static Optional<MirrorSession> createSession(ServerPlayer player, BlockPos sourcePos) {
        if (purgeMode) {
            InstantWorldMirror.LOGGER.warn("Cannot create session - purge mode is active");
            player.displayClientMessage((Component)Component.translatable((String)"message.instantworldmirror.purge_mode_active"), true);
            return Optional.empty();
        }
        sessionLock.writeLock().lock();
        try {
            MirrorSession session;
            int dimIndex;
            BlockPos playerStandPos;
            if (MirrorWorldManager.hasActiveSession(player.getUUID())) {
                InstantWorldMirror.LOGGER.warn("Player {} already has an active session", (Object)player.getName().getString());
                Optional<MirrorSession> optional = Optional.empty();
                return optional;
            }
            ServerLevel sourceLevel = (ServerLevel)player.level();
            boolean sourceInWater = MirrorWorldManager.isPositionInWater(sourceLevel, playerStandPos = sourcePos.above());
            if (sourceInWater) {
                InstantWorldMirror.LOGGER.info("Player {} creating mirror from underwater position at {}", (Object)player.getName().getString(), (Object)sourcePos);
            }
            if ((dimIndex = DimensionPool.allocateDimension((session = new MirrorSession(player.getUUID(), sourcePos, (ResourceKey<Level>)player.level().dimension(), sourceInWater)).getSessionId(), session.getSourceDimension())) < 0) {
                int poolSize = ModDimensions.getPoolSize();
                int inUse = DimensionPool.getInUseCount();
                int cleaning = DimensionPool.getCleaningCount();
                InstantWorldMirror.LOGGER.warn("No available dimensions for player {} (pool: {}, in use: {}, cleaning: {})", new Object[]{player.getName().getString(), poolSize, inUse, cleaning});
                player.displayClientMessage((Component)Component.translatable((String)"message.instantworldmirror.no_dimensions_available_detail", (Object[])new Object[]{poolSize, inUse, cleaning}), false);
                Optional<MirrorSession> optional = Optional.empty();
                return optional;
            }
            session.setDimensionIndex(dimIndex);
            activeSessions.put(session.getSessionId(), session);
            playerOwnedSession.put(player.getUUID(), session.getSessionId());
            InstantWorldMirror.LOGGER.debug("Created session {} for player {} using dimension {}", new Object[]{session.getSessionId(), player.getName().getString(), dimIndex});
            Optional<MirrorSession> optional = Optional.of(session);
            return optional;
        }
        finally {
            sessionLock.writeLock().unlock();
        }
    }

    public static void bindPortalToSession(UUID sessionId, UUID portalEntityId) {
        MirrorSession session = activeSessions.get(sessionId);
        if (session != null) {
            session.setPortalEntityId(portalEntityId);
            portalToSession.put(portalEntityId, sessionId);
            InstantWorldMirror.LOGGER.debug("Bound portal {} to session {}", (Object)portalEntityId, (Object)sessionId);
        }
    }

    public static boolean canAccessMirrorWorld(ServerPlayer player) {
        return playerAccessDenied.getOrDefault(player.getUUID(), false) == false;
    }

    public static void setAccessPermission(UUID playerId, boolean allowed) {
        if (allowed) {
            playerAccessDenied.remove(playerId);
        } else {
            playerAccessDenied.put(playerId, true);
        }
    }

    public static ResourceKey<Level> getPlayerOriginalDimension(ServerPlayer player) {
        return playerOriginalDimensions.get(player.getUUID());
    }

    public static boolean isBeingTeleportedByMod(UUID playerId) {
        return playersBeingTeleported.contains(playerId);
    }

    public static void markPlayerBeingTeleported(UUID playerId) {
        playersBeingTeleported.add(playerId);
    }

    public static void unmarkPlayerBeingTeleported(UUID playerId) {
        playersBeingTeleported.remove(playerId);
    }

    private static void cleanupPlayerTrackingData(UUID playerId) {
        playerOriginalPositions.remove(playerId);
        playerOriginalDimensions.remove(playerId);
        playerToSession.remove(playerId);
        playerItemTransferPermission.remove(playerId);
        playerOwnedSession.remove(playerId);
    }

    public static int prepareWorldCopy(ServerPlayer player, MirrorSession session) {
        if (player.level().isClientSide) {
            return 0;
        }
        MinecraftServer server = player.getServer();
        if (server == null) {
            return 0;
        }
        ServerLevel mirrorWorld = DimensionPool.getDimensionLevel(server, session.getDimensionIndex());
        if (mirrorWorld == null) {
            InstantWorldMirror.LOGGER.error("Mirror world dimension {} not found!", (Object)session.getDimensionIndex());
            return 0;
        }
        ServerLevel sourceWorld = server.getLevel(session.getSourceDimension());
        if (sourceWorld == null) {
            sourceWorld = server.overworld();
        }
        int queuePosition = WorldCopyService.queueWorldCopy(session, sourceWorld);
        InstantWorldMirror.LOGGER.debug("Queued world copy for session {} to dimension {}", (Object)session.getSessionId(), (Object)session.getDimensionIndex());
        return queuePosition;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean teleportToMirrorWorld(ServerPlayer player, MirrorSession session) {
        boolean allowWater;
        BlockPos baseTargetPos;
        ServerLevel mirrorWorld;
        if (player.level().isClientSide) {
            return false;
        }
        MinecraftServer server = player.getServer();
        if (server == null) {
            return false;
        }
        sessionLock.writeLock().lock();
        try {
            if (session.isDestroyed()) {
                InstantWorldMirror.LOGGER.warn("Cannot teleport to destroyed session {}", (Object)session.getSessionId());
                player.displayClientMessage((Component)Component.translatable((String)"message.instantworldmirror.session_expired"), true);
                boolean bl = false;
                return bl;
            }
            if (playerToSession.containsKey(player.getUUID())) {
                InstantWorldMirror.LOGGER.warn("Player {} is already in a session", (Object)player.getName().getString());
                player.displayClientMessage((Component)Component.translatable((String)"message.instantworldmirror.already_in_session"), true);
                boolean bl = false;
                return bl;
            }
            playerOriginalPositions.put(player.getUUID(), player.blockPosition());
            playerOriginalDimensions.put(player.getUUID(), (ResourceKey<Level>)player.level().dimension());
            ResourceKey<Level> mirrorDimKey = session.getMirrorDimension();
            if (mirrorDimKey == null) {
                InstantWorldMirror.LOGGER.error("Session {} has no allocated dimension!", (Object)session.getSessionId());
                boolean bl = false;
                return bl;
            }
            mirrorWorld = server.getLevel(mirrorDimKey);
            if (mirrorWorld == null) {
                InstantWorldMirror.LOGGER.error("Mirror world dimension {} not found!", mirrorDimKey);
                boolean bl = false;
                return bl;
            }
            session.addPlayer(player.getUUID());
            playerToSession.put(player.getUUID(), session.getSessionId());
            playerOwnedSession.remove(player.getUUID());
            baseTargetPos = session.getSourcePosition().above();
            allowWater = session.isSourceInWater();
        }
        finally {
            sessionLock.writeLock().unlock();
        }
        CompoundTag persistentData = player.getPersistentData();
        persistentData.putUUID(SESSION_ID_KEY, session.getSessionId());
        MirrorWorldManager.savePlayerInventory(player);
        BlockPos safePos = MirrorWorldManager.findSafeLandingPosition(mirrorWorld, baseTargetPos, allowWater);
        if (safePos == null) {
            safePos = baseTargetPos;
            MirrorWorldManager.clearAreaForPlayer(mirrorWorld, safePos);
            InstantWorldMirror.LOGGER.info("Cleared 1x2 area for player {} at {} in mirror world", (Object)player.getName().getString(), (Object)safePos);
            player.displayClientMessage((Component)Component.translatable((String)"message.instantworldmirror.area_cleared"), true);
        }
        player.teleportTo(mirrorWorld, (double)safePos.getX() + 0.5, (double)safePos.getY(), (double)safePos.getZ() + 0.5, player.getYRot(), player.getXRot());
        CompoundTag playerData = player.getPersistentData();
        playerData.putLong("MirrorWorldEnterTime", mirrorWorld.getGameTime());
        MirrorWorldManager.syncDimensionEffectsToPlayer(player, session);
        if (session.isHost(player.getUUID())) {
            MirrorWorldManager.spawnReturnPortal(mirrorWorld, player, safePos, session.getSessionId());
        }
        InstantWorldMirror.LOGGER.info("Player {} teleported to Mirror World dimension {} (session: {}, players: {})", new Object[]{player.getName().getString(), session.getDimensionIndex(), session.getSessionId(), session.getPlayerCount()});
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static boolean returnToOverworld(ServerPlayer player) {
        boolean allowItemTransfer;
        BlockPos targetPos;
        ServerLevel targetLevel;
        MirrorSession session;
        if (player.level().isClientSide) {
            return false;
        }
        MinecraftServer server = player.getServer();
        if (server == null) {
            return false;
        }
        if (!MirrorWorldManager.isInMirrorWorld(player)) {
            InstantWorldMirror.LOGGER.warn("returnToOverworld: player {} is not in mirror world", (Object)player.getName().getString());
            return false;
        }
        sessionLock.readLock().lock();
        try {
            UUID sessionId = playerToSession.get(player.getUUID());
            session = sessionId != null ? activeSessions.get(sessionId) : null;
        }
        finally {
            sessionLock.readLock().unlock();
        }
        BlockPos originalPos = playerOriginalPositions.get(player.getUUID());
        ResourceKey originalDimension = playerOriginalDimensions.get(player.getUUID());
        if (originalPos == null || originalDimension == null) {
            ResourceLocation dimLoc;
            CompoundTag persistentData = player.getPersistentData();
            if (persistentData.contains("instantworldmirror_original_pos_x")) {
                originalPos = new BlockPos(persistentData.getInt("instantworldmirror_original_pos_x"), persistentData.getInt("instantworldmirror_original_pos_y"), persistentData.getInt("instantworldmirror_original_pos_z"));
            }
            if (persistentData.contains(ORIGINAL_DIM_KEY) && (dimLoc = ResourceLocation.tryParse((String)persistentData.getString(ORIGINAL_DIM_KEY))) != null) {
                originalDimension = ResourceKey.create((ResourceKey)Registries.DIMENSION, (ResourceLocation)dimLoc);
            }
            if (originalPos != null && originalDimension != null) {
                InstantWorldMirror.LOGGER.info("Restored return position from persistent data for {}", (Object)player.getName().getString());
            }
        }
        if (originalPos != null && originalDimension != null) {
            targetLevel = server.getLevel(originalDimension);
            targetPos = originalPos;
        } else {
            targetLevel = server.overworld();
            targetPos = targetLevel.getSharedSpawnPos();
            InstantWorldMirror.LOGGER.warn("No saved position for {}, using spawn", (Object)player.getName().getString());
        }
        if (targetLevel == null) {
            targetLevel = server.overworld();
            targetPos = targetLevel.getSharedSpawnPos();
        }
        if (allowItemTransfer = playerItemTransferPermission.getOrDefault(player.getUUID(), false).booleanValue()) {
            MirrorWorldManager.clearSavedData(player);
        } else {
            MirrorWorldManager.restorePlayerInventory(player);
        }
        playersBeingTeleported.add(player.getUUID());
        try {
            player.teleportTo(targetLevel, (double)targetPos.getX() + 0.5, (double)targetPos.getY(), (double)targetPos.getZ() + 0.5, player.getYRot(), player.getXRot());
        }
        finally {
            playersBeingTeleported.remove(player.getUUID());
        }
        if (MirrorWorldManager.isInMirrorWorld(player)) {
            InstantWorldMirror.LOGGER.error("Teleport FAILED - {} still in mirror world!", (Object)player.getName().getString());
            return false;
        }
        if (session != null) {
            MirrorWorldManager.clearDimensionEffectsForPlayer(player, session.getDimensionIndex());
        }
        sessionLock.writeLock().lock();
        try {
            MirrorWorldManager.cleanupPlayerTrackingData(player.getUUID());
            if (session != null) {
                boolean isHost = session.isHost(player.getUUID());
                if (isHost) {
                    InstantWorldMirror.LOGGER.info("Host {} leaving session {}, kicking all other players", (Object)player.getName().getString(), (Object)session.getSessionId());
                    MirrorWorldManager.kickAllPlayersFromSession(session, server, player.getUUID());
                    MirrorWorldManager.destroySession(session, server);
                } else {
                    boolean sessionNowEmpty = session.removePlayer(player.getUUID());
                    if (sessionNowEmpty) {
                        MirrorWorldManager.destroySession(session, server);
                    }
                }
            }
        }
        finally {
            sessionLock.writeLock().unlock();
        }
        player.displayClientMessage((Component)Component.translatable((String)"message.instantworldmirror.returned"), true);
        InstantWorldMirror.LOGGER.debug("Player {} returned from mirror world", (Object)player.getName().getString());
        return true;
    }

    public static boolean returnToOverworldFromPosition(ServerPlayer player, BlockPos portalPos) {
        if (player.level().isClientSide) {
            return false;
        }
        MinecraftServer server = player.getServer();
        if (server == null) {
            return false;
        }
        BlockPos originalPos = playerOriginalPositions.get(player.getUUID());
        if (originalPos == null || portalPos.closerThan((Vec3i)originalPos, 16.0)) {
            return MirrorWorldManager.returnToOverworld(player);
        }
        InstantWorldMirror.LOGGER.info("Player {} using distant return portal at {} (original entry: {}), teleporting to corresponding overworld position", new Object[]{player.getName().getString(), portalPos, originalPos});
        return MirrorWorldManager.returnToOverworldAtPosition(player, portalPos);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static boolean returnToOverworldAtPosition(ServerPlayer player, BlockPos targetMirrorPos) {
        MinecraftServer server = player.getServer();
        if (server == null) {
            return false;
        }
        if (!MirrorWorldManager.isInMirrorWorld(player)) {
            InstantWorldMirror.LOGGER.warn("returnToOverworldAtPosition: player {} is not in mirror world", (Object)player.getName().getString());
            return false;
        }
        sessionLock.writeLock().lock();
        try {
            ServerLevel targetLevel;
            ResourceLocation dimLoc;
            CompoundTag persistentData;
            UUID sessionId = playerToSession.get(player.getUUID());
            MirrorSession session = sessionId != null ? activeSessions.get(sessionId) : null;
            ResourceKey originalDimension = playerOriginalDimensions.get(player.getUUID());
            if (originalDimension == null && (persistentData = player.getPersistentData()).contains(ORIGINAL_DIM_KEY) && (dimLoc = ResourceLocation.tryParse((String)persistentData.getString(ORIGINAL_DIM_KEY))) != null) {
                originalDimension = ResourceKey.create((ResourceKey)Registries.DIMENSION, (ResourceLocation)dimLoc);
            }
            if (originalDimension == null) {
                originalDimension = Level.OVERWORLD;
            }
            if ((targetLevel = server.getLevel(originalDimension)) == null) {
                targetLevel = server.overworld();
            }
            boolean allowWater = session != null && session.isSourceInWater();
            BlockPos safePos = MirrorWorldManager.findSafeLandingPosition(targetLevel, targetMirrorPos, allowWater);
            InstantWorldMirror.LOGGER.info("Found safe landing position {} for player {} (original target: {})", new Object[]{safePos, player.getName().getString(), targetMirrorPos});
            boolean allowItemTransfer = playerItemTransferPermission.getOrDefault(player.getUUID(), false);
            if (allowItemTransfer) {
                MirrorWorldManager.clearSavedData(player);
            } else {
                MirrorWorldManager.restorePlayerInventory(player);
            }
            playersBeingTeleported.add(player.getUUID());
            try {
                player.teleportTo(targetLevel, (double)safePos.getX() + 0.5, (double)safePos.getY(), (double)safePos.getZ() + 0.5, player.getYRot(), player.getXRot());
            }
            finally {
                playersBeingTeleported.remove(player.getUUID());
            }
            if (MirrorWorldManager.isInMirrorWorld(player)) {
                InstantWorldMirror.LOGGER.error("Teleport FAILED - {} still in mirror world!", (Object)player.getName().getString());
                boolean bl = false;
                return bl;
            }
            if (session != null) {
                MirrorWorldManager.clearDimensionEffectsForPlayer(player, session.getDimensionIndex());
            }
            MirrorWorldManager.cleanupPlayerTrackingData(player.getUUID());
            if (session != null) {
                boolean isHost = session.isHost(player.getUUID());
                if (isHost) {
                    InstantWorldMirror.LOGGER.info("Host {} leaving session {}, kicking all other players", (Object)player.getName().getString(), (Object)session.getSessionId());
                    MirrorWorldManager.kickAllPlayersFromSession(session, server, player.getUUID());
                    MirrorWorldManager.destroySession(session, server);
                } else {
                    boolean sessionNowEmpty = session.removePlayer(player.getUUID());
                    if (sessionNowEmpty) {
                        MirrorWorldManager.destroySession(session, server);
                    }
                }
            }
            player.displayClientMessage((Component)Component.translatable((String)"message.instantworldmirror.returned_to_corresponding_position"), true);
            InstantWorldMirror.LOGGER.info("Player {} returned from mirror world to corresponding position {}", (Object)player.getName().getString(), (Object)safePos);
            boolean bl = true;
            return bl;
        }
        finally {
            sessionLock.writeLock().unlock();
        }
    }

    public static boolean teleportToMirrorSpawn(ServerPlayer player) {
        int spawnY;
        BlockPos spawnPos;
        MirrorSession session;
        if (player.level().isClientSide) {
            return false;
        }
        if (!MirrorWorldManager.isInMirrorWorld(player)) {
            InstantWorldMirror.LOGGER.warn("teleportToMirrorSpawn: player {} is not in mirror world", (Object)player.getName().getString());
            player.displayClientMessage((Component)Component.translatable((String)"message.instantworldmirror.not_in_mirror_world"), true);
            return false;
        }
        UUID sessionId = playerToSession.get(player.getUUID());
        MirrorSession mirrorSession = session = sessionId != null ? activeSessions.get(sessionId) : null;
        if (session == null) {
            InstantWorldMirror.LOGGER.warn("teleportToMirrorSpawn: player {} has no active session", (Object)player.getName().getString());
            player.displayClientMessage((Component)Component.translatable((String)"message.instantworldmirror.no_active_session"), true);
            return false;
        }
        ServerLevel mirrorLevel = (ServerLevel)player.level();
        BlockPos safePos = MirrorWorldManager.findSafeSpawnNearbyPosition(mirrorLevel, spawnPos = session.getSourcePosition().above(), 2, 10, spawnY = spawnPos.getY());
        if (safePos == null) {
            InstantWorldMirror.LOGGER.info("No safe spawn position found for {}, returning to overworld", (Object)player.getName().getString());
            player.displayClientMessage((Component)Component.translatable((String)"message.instantworldmirror.no_safe_spawn_returning"), true);
            return MirrorWorldManager.returnToOverworld(player);
        }
        player.teleportTo((double)safePos.getX() + 0.5, (double)safePos.getY(), (double)safePos.getZ() + 0.5);
        player.level().playSound(null, player.getX(), player.getY(), player.getZ(), SoundEvents.ENDERMAN_TELEPORT, SoundSource.PLAYERS, 0.8f, 1.0f);
        if (!player.isCreative()) {
            int baseCooldownSeconds = MirrorConfig.getMirrorCooldownTicks() / 20;
            int finalCooldownSeconds = Math.max(30, baseCooldownSeconds);
            DimensionMirrorItem.setCooldown(player.getUUID(), finalCooldownSeconds);
            DimensionMirrorItem.syncCooldownToClient(player);
            InstantWorldMirror.LOGGER.debug("Applied cooldown {} seconds to {} for teleport to spawn", (Object)finalCooldownSeconds, (Object)player.getName().getString());
        }
        player.displayClientMessage((Component)Component.translatable((String)"message.instantworldmirror.teleported_to_spawn"), true);
        InstantWorldMirror.LOGGER.info("Player {} teleported to mirror spawn at ({}, {}, {})", new Object[]{player.getName().getString(), safePos.getX(), safePos.getY(), safePos.getZ()});
        return true;
    }

    private static BlockPos findSafeSpawnNearbyPosition(ServerLevel level, BlockPos spawnPos, int minRadius, int maxRadius, int targetY) {
        for (int radius = minRadius; radius <= maxRadius; ++radius) {
            for (int dx = -radius; dx <= radius; ++dx) {
                for (int dz = -radius; dz <= radius; ++dz) {
                    BlockPos checkPos;
                    double horizontalDistance = Math.sqrt(dx * dx + dz * dz);
                    if (horizontalDistance < (double)minRadius || horizontalDistance > (double)maxRadius || Math.abs(horizontalDistance - (double)radius) > 0.5 && radius < maxRadius || !MirrorWorldManager.isValidSpawnNearbyPosition(level, checkPos = new BlockPos(spawnPos.getX() + dx, targetY, spawnPos.getZ() + dz), spawnPos)) continue;
                    return checkPos;
                }
            }
        }
        return null;
    }

    private static boolean isValidSpawnNearbyPosition(ServerLevel level, BlockPos pos, BlockPos spawnPos) {
        if (pos.getY() < level.getMinBuildHeight() + 1 || pos.getY() > level.getMaxBuildHeight() - 2) {
            return false;
        }
        BlockPos below = pos.below();
        BlockPos above = pos.above();
        BlockState belowState = level.getBlockState(below);
        BlockState atState = level.getBlockState(pos);
        BlockState aboveState = level.getBlockState(above);
        if (!belowState.isSolid()) {
            return false;
        }
        if (MirrorWorldManager.isDangerousBlock(belowState)) {
            return false;
        }
        if (atState.isSolid() || aboveState.isSolid()) {
            return false;
        }
        if (atState.getFluidState().isSource() || aboveState.getFluidState().isSource()) {
            return false;
        }
        return level.canSeeSky(pos);
    }

    private static boolean isPositionInWater(ServerLevel level, BlockPos pos) {
        BlockState state = level.getBlockState(pos);
        return state.getFluidState().isSource() && state.getFluidState().getType() == Fluids.WATER;
    }

    private static BlockPos findSafeLandingPosition(ServerLevel level, BlockPos targetPos, boolean allowWater) {
        if (MirrorWorldManager.isPositionSafe(level, targetPos, allowWater)) {
            return targetPos;
        }
        int maxRadius = 16;
        for (int radius = 1; radius <= maxRadius; ++radius) {
            for (int dx = -radius; dx <= radius; ++dx) {
                for (int dz = -radius; dz <= radius; ++dz) {
                    BlockPos checkPos;
                    BlockPos safePos;
                    if (Math.abs(dx) != radius && Math.abs(dz) != radius || (safePos = MirrorWorldManager.findSafeYLevel(level, checkPos = targetPos.offset(dx, 0, dz), allowWater)) == null || !MirrorWorldManager.isPositionSafe(level, safePos, allowWater)) continue;
                    return safePos;
                }
            }
        }
        BlockPos verticalSafe = MirrorWorldManager.findSafeYLevel(level, targetPos, allowWater);
        if (verticalSafe != null) {
            return verticalSafe;
        }
        return null;
    }

    private static BlockPos findSafeYLevel(ServerLevel level, BlockPos horizontalPos, boolean allowWater) {
        BlockPos checkPos;
        int y;
        int minY = level.getMinBuildHeight();
        int maxY = level.getMaxBuildHeight();
        int startY = Math.max(minY + 1, Math.min(horizontalPos.getY(), maxY - 3));
        int maxVerticalSearch = 5;
        int searchMinY = Math.max(minY + 1, startY - maxVerticalSearch);
        int searchMaxY = Math.min(maxY - 2, startY + maxVerticalSearch);
        for (y = startY; y >= searchMinY; --y) {
            checkPos = new BlockPos(horizontalPos.getX(), y, horizontalPos.getZ());
            if (!MirrorWorldManager.isPositionSafe(level, checkPos, allowWater)) continue;
            return checkPos;
        }
        for (y = startY + 1; y <= searchMaxY; ++y) {
            checkPos = new BlockPos(horizontalPos.getX(), y, horizontalPos.getZ());
            if (!MirrorWorldManager.isPositionSafe(level, checkPos, allowWater)) continue;
            return checkPos;
        }
        return null;
    }

    private static boolean isPositionSafe(ServerLevel level, BlockPos pos, boolean allowWater) {
        if (pos.getY() < level.getMinBuildHeight() + 1 || pos.getY() > level.getMaxBuildHeight() - 2) {
            return false;
        }
        BlockPos below = pos.below();
        BlockPos above = pos.above();
        BlockState belowState = level.getBlockState(below);
        BlockState atState = level.getBlockState(pos);
        BlockState aboveState = level.getBlockState(above);
        if (allowWater && atState.getFluidState().isSource() && atState.getFluidState().getType() == Fluids.WATER && !atState.isSolid() && !aboveState.isSolid() && !MirrorWorldManager.isDangerousBlock(atState) && !MirrorWorldManager.isDangerousBlock(aboveState)) {
            return true;
        }
        if (!belowState.isSolid()) {
            return false;
        }
        if (MirrorWorldManager.isDangerousBlock(belowState)) {
            return false;
        }
        if (atState.isSolid() || aboveState.isSolid()) {
            return false;
        }
        if (MirrorWorldManager.isDangerousBlock(atState) || MirrorWorldManager.isDangerousBlock(aboveState)) {
            return false;
        }
        if (atState.getFluidState().isSource() || aboveState.getFluidState().isSource()) {
            return false;
        }
        return !MirrorWorldManager.hasNearbyPortal(level, pos);
    }

    private static boolean hasNearbyPortal(ServerLevel level, BlockPos pos) {
        double checkRadius = 2.5;
        AABB searchArea = new AABB((double)pos.getX() - checkRadius, (double)(pos.getY() - 1), (double)pos.getZ() - checkRadius, (double)pos.getX() + checkRadius, (double)(pos.getY() + 3), (double)pos.getZ() + checkRadius);
        List nearbyPortals = level.getEntitiesOfClass(MirrorPortalEntity.class, searchArea);
        return !nearbyPortals.isEmpty();
    }

    private static boolean isDangerousBlock(BlockState state) {
        Block block = state.getBlock();
        if (block == Blocks.LAVA) {
            return true;
        }
        if (block == Blocks.FIRE || block == Blocks.SOUL_FIRE) {
            return true;
        }
        if (block == Blocks.MAGMA_BLOCK) {
            return true;
        }
        if (block == Blocks.CACTUS) {
            return true;
        }
        if (block == Blocks.SWEET_BERRY_BUSH) {
            return true;
        }
        if (block == Blocks.WITHER_ROSE) {
            return true;
        }
        return block == Blocks.POWDER_SNOW;
    }

    private static void clearAreaForPlayer(ServerLevel level, BlockPos pos) {
        BlockPos abovePos = pos.above();
        level.setBlock(pos, Blocks.AIR.defaultBlockState(), 3);
        level.setBlock(abovePos, Blocks.AIR.defaultBlockState(), 3);
        BlockPos belowPos = pos.below();
        BlockState belowState = level.getBlockState(belowPos);
        if (!belowState.isSolid() || MirrorWorldManager.isDangerousBlock(belowState)) {
            level.setBlock(belowPos, Blocks.STONE.defaultBlockState(), 3);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void kickAllPlayersFromSession(MirrorSession session, MinecraftServer server, UUID excludePlayerId) {
        HashSet<UUID> playersToKick = new HashSet<UUID>(session.getPlayers());
        playersToKick.remove(excludePlayerId);
        if (playersToKick.isEmpty()) {
            return;
        }
        InstantWorldMirror.LOGGER.info("Kicking {} players from session {} because host left", (Object)playersToKick.size(), (Object)session.getSessionId());
        ServerLevel overworld = server.overworld();
        BlockPos spawnPos = overworld.getSharedSpawnPos();
        for (UUID playerId : playersToKick) {
            ServerLevel level;
            ServerPlayer kickedPlayer = server.getPlayerList().getPlayer(playerId);
            if (kickedPlayer == null) {
                MirrorWorldManager.cleanupPlayerTrackingData(playerId);
                session.removePlayer(playerId);
                continue;
            }
            MirrorWorldManager.restorePlayerInventory(kickedPlayer);
            BlockPos originalPos = playerOriginalPositions.get(playerId);
            ResourceKey<Level> originalDim = playerOriginalDimensions.get(playerId);
            ServerLevel targetLevel = overworld;
            BlockPos targetPos = spawnPos;
            if (originalPos != null && originalDim != null && (level = server.getLevel(originalDim)) != null) {
                targetLevel = level;
                targetPos = originalPos;
            }
            playersBeingTeleported.add(playerId);
            try {
                kickedPlayer.teleportTo(targetLevel, (double)targetPos.getX() + 0.5, (double)targetPos.getY(), (double)targetPos.getZ() + 0.5, kickedPlayer.getYRot(), kickedPlayer.getXRot());
            }
            finally {
                playersBeingTeleported.remove(playerId);
            }
            MirrorWorldManager.clearDimensionEffectsForPlayer(kickedPlayer, session.getDimensionIndex());
            MirrorWorldManager.cleanupPlayerTrackingData(playerId);
            session.removePlayer(playerId);
            kickedPlayer.displayClientMessage((Component)Component.translatable((String)"message.instantworldmirror.host_left"), true);
            InstantWorldMirror.LOGGER.info("Kicked player {} from session because host left", (Object)kickedPlayer.getName().getString());
        }
    }

    private static void destroySession(MirrorSession session, MinecraftServer server) {
        int dimIndex;
        if (session.isDestroyed()) {
            return;
        }
        session.markDestroyed();
        UUID sessionId = session.getSessionId();
        activeSessions.remove(sessionId);
        playerOwnedSession.values().removeIf(id -> id.equals(sessionId));
        UUID portalId = session.getPortalEntityId();
        if (portalId != null) {
            Entity portalEntity;
            portalToSession.remove(portalId);
            ServerLevel sourceLevel = server.getLevel(session.getSourceDimension());
            if (sourceLevel != null && (portalEntity = sourceLevel.getEntity(portalId)) != null) {
                portalEntity.discard();
                InstantWorldMirror.LOGGER.debug("Destroyed portal entity {} for session {}", (Object)portalId, (Object)sessionId);
            }
        }
        if ((dimIndex = session.getDimensionIndex()) >= 0) {
            DimensionPool.releaseDimension(sessionId);
            ServerLevel mirrorWorld = server.getLevel(session.getMirrorDimension());
            if (mirrorWorld != null) {
                WorldCopyService.cleanupMirrorWorld(mirrorWorld, dimIndex);
            }
        }
        InstantWorldMirror.LOGGER.debug("Session {} destroyed, dimension {} queued for cleanup", (Object)sessionId, (Object)dimIndex);
    }

    public static void cancelSession(UUID sessionId, MinecraftServer server) {
        sessionLock.writeLock().lock();
        try {
            MirrorSession session = activeSessions.get(sessionId);
            if (session != null) {
                playerOwnedSession.remove(session.getCreatorId());
                if (session.isEmpty()) {
                    MirrorWorldManager.destroySession(session, server);
                } else {
                    InstantWorldMirror.LOGGER.info("Session {} portal timed out but has {} players, keeping session", (Object)sessionId, (Object)session.getPlayerCount());
                }
            }
        }
        finally {
            sessionLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void handlePlayerDisconnect(ServerPlayer player, MinecraftServer server) {
        sessionLock.writeLock().lock();
        try {
            boolean sessionNowEmpty;
            MirrorSession session;
            UUID sessionId;
            MirrorSession session2;
            UUID playerId = player.getUUID();
            UUID ownedSessionId = playerOwnedSession.remove(playerId);
            if (ownedSessionId != null && (session2 = activeSessions.get(ownedSessionId)) != null && session2.isEmpty()) {
                MirrorWorldManager.destroySession(session2, server);
            }
            if ((sessionId = playerToSession.get(playerId)) != null && (session = activeSessions.get(sessionId)) != null && (sessionNowEmpty = session.removePlayer(playerId))) {
                MirrorWorldManager.destroySession(session, server);
            }
            MirrorWorldManager.cleanupPlayerTrackingData(playerId);
        }
        finally {
            sessionLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void handleMirrorWorldDeath(ServerPlayer player, MinecraftServer server) {
        sessionLock.writeLock().lock();
        try {
            UUID playerId = player.getUUID();
            UUID sessionId = playerToSession.get(playerId);
            MirrorSession session = sessionId != null ? activeSessions.get(sessionId) : null;
            MirrorWorldManager.restorePlayerInventory(player);
            if (session != null) {
                MirrorWorldManager.clearDimensionEffectsForPlayer(player, session.getDimensionIndex());
                boolean sessionNowEmpty = session.removePlayer(playerId);
                InstantWorldMirror.LOGGER.info("Player {} died and left session {} (players remaining: {})", new Object[]{player.getName().getString(), sessionId, session.getPlayerCount()});
                if (sessionNowEmpty) {
                    MirrorWorldManager.destroySession(session, server);
                }
            }
            MirrorWorldManager.cleanupPlayerTrackingData(playerId);
            InstantWorldMirror.LOGGER.info("Cleaned up mirror world session data for {} after death", (Object)player.getName().getString());
        }
        finally {
            sessionLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void handleExternalExit(ServerPlayer player, MinecraftServer server) {
        sessionLock.writeLock().lock();
        try {
            UUID playerId = player.getUUID();
            UUID sessionId = playerToSession.get(playerId);
            if (sessionId != null) {
                MirrorSession session = activeSessions.get(sessionId);
                if (session != null) {
                    InstantWorldMirror.LOGGER.info("Player {} externally left session {}", (Object)player.getName().getString(), (Object)sessionId);
                    boolean allowItemTransfer = playerItemTransferPermission.getOrDefault(playerId, false);
                    if (allowItemTransfer) {
                        MirrorWorldManager.clearSavedData(player);
                    } else {
                        MirrorWorldManager.restorePlayerInventory(player);
                    }
                    boolean sessionNowEmpty = session.removePlayer(playerId);
                    if (sessionNowEmpty) {
                        MirrorWorldManager.destroySession(session, server);
                    }
                }
                player.displayClientMessage((Component)Component.translatable((String)"message.instantworldmirror.external_exit"), false);
            }
            MirrorWorldManager.cleanupPlayerTrackingData(playerId);
        }
        finally {
            sessionLock.writeLock().unlock();
        }
    }

    private static void spawnReturnPortal(ServerLevel mirrorWorld, ServerPlayer player, BlockPos targetPos, UUID sessionId) {
        BlockPos groundBlockPos = targetPos.below();
        mirrorWorld.playSound(null, groundBlockPos, SoundEvents.PORTAL_TRIGGER, SoundSource.PLAYERS, 0.5f, 1.2f);
        MirrorWorldManager.spawnPortalParticles(mirrorWorld, groundBlockPos);
        MirrorPortalEntity.spawnReturnPortal(mirrorWorld, groundBlockPos, player.getUUID(), true, sessionId);
        int dimIndex = ModDimensions.getMirrorWorldIndex((ResourceKey<Level>)mirrorWorld.dimension());
        if (dimIndex >= 0) {
            WorldCopyService.trackModifiedChunk(dimIndex, groundBlockPos.getX() >> 4, groundBlockPos.getZ() >> 4);
        }
    }

    private static void spawnPortalParticles(ServerLevel level, BlockPos pos) {
        double x = (double)pos.getX() + 0.5;
        double y = (double)pos.getY() + 1.0;
        double z = (double)pos.getZ() + 0.5;
        for (int i = 0; i < 20; ++i) {
            double offsetX = (level.random.nextDouble() - 0.5) * 2.0;
            double offsetY = level.random.nextDouble() * 2.0;
            double offsetZ = (level.random.nextDouble() - 0.5) * 2.0;
            level.sendParticles((ParticleOptions)ParticleTypes.PORTAL, x + offsetX, y + offsetY, z + offsetZ, 1, 0.0, 0.0, 0.0, 0.0);
        }
    }

    public static boolean isInMirrorWorld(ServerPlayer player) {
        return ModDimensions.isAnyMirrorWorld((ResourceKey<Level>)player.level().dimension());
    }

    public static void setItemTransferPermission(UUID playerId, boolean allowed) {
        playerItemTransferPermission.put(playerId, allowed);
    }

    public static boolean getItemTransferPermission(UUID playerId) {
        return playerItemTransferPermission.getOrDefault(playerId, (Boolean)MirrorConfig.ALLOW_ITEM_TRANSFER.get());
    }

    public static boolean forceReturn(ServerPlayer player) {
        return MirrorWorldManager.returnToOverworld(player);
    }

    public static int getActiveSessionCount() {
        return activeSessions.size();
    }

    public static int getTotalPlayersInMirrorWorld() {
        return playerToSession.size();
    }

    private static void syncDimensionEffectsToPlayer(ServerPlayer player, MirrorSession session) {
        ResourceKey<Level> sourceDim = session.getSourceDimension();
        int dimIndex = session.getDimensionIndex();
        ServerLevel sourceLevel = player.getServer().getLevel(sourceDim);
        ResourceLocation effectsLoc = sourceLevel != null ? sourceLevel.dimensionType().effectsLocation() : ResourceLocation.withDefaultNamespace((String)"overworld");
        SyncMirrorEffectsPacket packet = new SyncMirrorEffectsPacket(dimIndex, effectsLoc.toString());
        PacketDistributor.sendToPlayer((ServerPlayer)player, (CustomPacketPayload)packet, (CustomPacketPayload[])new CustomPacketPayload[0]);
        InstantWorldMirror.LOGGER.debug("Synced mirror effects to {}: dim {} -> {}", new Object[]{player.getName().getString(), dimIndex, effectsLoc});
    }

    private static void clearDimensionEffectsForPlayer(ServerPlayer player, int dimIndex) {
        ClearMirrorEffectsPacket packet = new ClearMirrorEffectsPacket(dimIndex);
        PacketDistributor.sendToPlayer((ServerPlayer)player, (CustomPacketPayload)packet, (CustomPacketPayload[])new CustomPacketPayload[0]);
        InstantWorldMirror.LOGGER.debug("Cleared mirror effects for {}: dim {}", (Object)player.getName().getString(), (Object)dimIndex);
    }

    private static void savePlayerInventory(ServerPlayer player) {
        if (!player.gameMode.isSurvival()) {
            InstantWorldMirror.LOGGER.info("Skipping inventory save for non-survival player {}", (Object)player.getName().getString());
            return;
        }
        CompoundTag persistentData = player.getPersistentData();
        ListTag inventoryTag = new ListTag();
        player.getInventory().save(inventoryTag);
        persistentData.put(SAVED_INVENTORY_KEY, (Tag)inventoryTag);
        ListTag enderChestTag = player.getEnderChestInventory().createTag((HolderLookup.Provider)player.registryAccess());
        persistentData.put(SAVED_ENDERCHEST_KEY, (Tag)enderChestTag);
        BlockPos pos = player.blockPosition();
        persistentData.putInt("instantworldmirror_original_pos_x", pos.getX());
        persistentData.putInt("instantworldmirror_original_pos_y", pos.getY());
        persistentData.putInt("instantworldmirror_original_pos_z", pos.getZ());
        persistentData.putString(ORIGINAL_DIM_KEY, player.level().dimension().location().toString());
        InstantWorldMirror.LOGGER.info("Saved inventory and ender chest to persistent data for player {} ({} inventory items, {} ender chest items)", new Object[]{player.getName().getString(), inventoryTag.size(), enderChestTag.size()});
    }

    private static void restorePlayerInventory(ServerPlayer player) {
        if (!player.gameMode.isSurvival()) {
            InstantWorldMirror.LOGGER.info("Skipping inventory restore for non-survival player {}", (Object)player.getName().getString());
            MirrorWorldManager.clearSavedData(player);
            return;
        }
        CompoundTag persistentData = player.getPersistentData();
        if (persistentData.contains(SAVED_INVENTORY_KEY)) {
            ListTag savedInventory = persistentData.getList(SAVED_INVENTORY_KEY, 10);
            player.getInventory().clearContent();
            player.getInventory().load(savedInventory);
            if (persistentData.contains(SAVED_ENDERCHEST_KEY)) {
                ListTag savedEnderChest = persistentData.getList(SAVED_ENDERCHEST_KEY, 10);
                player.getEnderChestInventory().clearContent();
                player.getEnderChestInventory().fromTag(savedEnderChest, (HolderLookup.Provider)player.registryAccess());
                InstantWorldMirror.LOGGER.info("Restored inventory and ender chest from persistent data for player {}", (Object)player.getName().getString());
            } else {
                player.getEnderChestInventory().clearContent();
                InstantWorldMirror.LOGGER.info("Restored inventory from persistent data for player {} (cleared ender chest - no saved data)", (Object)player.getName().getString());
            }
            MirrorWorldManager.clearSavedData(player);
        } else {
            InstantWorldMirror.LOGGER.warn("No saved inventory found in persistent data for player {}", (Object)player.getName().getString());
        }
    }

    public static boolean hasSavedInventory(ServerPlayer player) {
        return player.getPersistentData().contains(SAVED_INVENTORY_KEY);
    }

    public static void clearSavedData(ServerPlayer player) {
        CompoundTag persistentData = player.getPersistentData();
        persistentData.remove(SAVED_INVENTORY_KEY);
        persistentData.remove(SAVED_ENDERCHEST_KEY);
        persistentData.remove("instantworldmirror_original_pos_x");
        persistentData.remove("instantworldmirror_original_pos_y");
        persistentData.remove("instantworldmirror_original_pos_z");
        persistentData.remove(ORIGINAL_DIM_KEY);
        persistentData.remove(SESSION_ID_KEY);
        InstantWorldMirror.LOGGER.info("Cleared saved data for player {}", (Object)player.getName().getString());
    }

    public static int getSessionPlayerCount(UUID sessionId) {
        sessionLock.readLock().lock();
        try {
            MirrorSession session = activeSessions.get(sessionId);
            if (session != null) {
                int n = session.getPlayerCount();
                return n;
            }
            int n = 0;
            return n;
        }
        finally {
            sessionLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static int forceClearDimension(int dimIndex, MinecraftServer server) {
        sessionLock.writeLock().lock();
        try {
            Optional<UUID> sessionIdOpt;
            int playersReturned = 0;
            WorldCopyService.cancelCleanupTask(dimIndex);
            ServerLevel mirrorWorld = DimensionPool.getDimensionLevel(server, dimIndex);
            ResourceKey<Level> mirrorDimKey = ModDimensions.getMirrorWorld(dimIndex);
            HashSet<ServerPlayer> allPlayersToMove = new HashSet<ServerPlayer>();
            if (mirrorWorld != null) {
                allPlayersToMove.addAll(mirrorWorld.players());
            }
            for (ServerPlayer serverPlayer : server.getPlayerList().getPlayers()) {
                if (!serverPlayer.level().dimension().equals(mirrorDimKey)) continue;
                allPlayersToMove.add(serverPlayer);
            }
            if (!allPlayersToMove.isEmpty()) {
                ServerLevel overworld = server.overworld();
                BlockPos defaultSpawnPoint = overworld.getSharedSpawnPos();
                for (ServerPlayer player : allPlayersToMove) {
                    BlockPos targetPos;
                    ServerLevel targetLevel;
                    UUID playerId = player.getUUID();
                    BlockPos originalPos = playerOriginalPositions.get(playerId);
                    ResourceKey originalDimension = playerOriginalDimensions.get(playerId);
                    if (originalPos == null || originalDimension == null) {
                        String dimString;
                        ResourceLocation dimLoc;
                        CompoundTag persistentData = player.getPersistentData();
                        if (persistentData.contains("instantworldmirror_original_pos_x")) {
                            int x = persistentData.getInt("instantworldmirror_original_pos_x");
                            int y = persistentData.getInt("instantworldmirror_original_pos_y");
                            int z = persistentData.getInt("instantworldmirror_original_pos_z");
                            originalPos = new BlockPos(x, y, z);
                        }
                        if (persistentData.contains(ORIGINAL_DIM_KEY) && (dimLoc = ResourceLocation.tryParse((String)(dimString = persistentData.getString(ORIGINAL_DIM_KEY)))) != null) {
                            originalDimension = ResourceKey.create((ResourceKey)Registries.DIMENSION, (ResourceLocation)dimLoc);
                        }
                        if (originalPos != null && originalDimension != null) {
                            playerOriginalPositions.put(playerId, originalPos);
                            playerOriginalDimensions.put(playerId, (ResourceKey<Level>)originalDimension);
                        }
                    }
                    if (originalPos != null && originalDimension != null) {
                        targetLevel = server.getLevel(originalDimension);
                        targetPos = originalPos;
                    } else {
                        targetLevel = overworld;
                        targetPos = defaultSpawnPoint;
                        originalDimension = Level.OVERWORLD;
                    }
                    if (targetLevel == null) {
                        targetLevel = overworld;
                        targetPos = defaultSpawnPoint;
                        originalDimension = Level.OVERWORLD;
                    }
                    playerOriginalDimensions.put(playerId, (ResourceKey<Level>)originalDimension);
                    playersBeingTeleported.add(playerId);
                    try {
                        player.teleportTo(targetLevel, (double)targetPos.getX() + 0.5, (double)targetPos.getY(), (double)targetPos.getZ() + 0.5, Set.of(), player.getYRot(), player.getXRot());
                    }
                    finally {
                        playersBeingTeleported.remove(playerId);
                    }
                    MirrorWorldManager.clearSavedData(player);
                    MirrorWorldManager.cleanupPlayerTrackingData(playerId);
                    player.displayClientMessage((Component)Component.translatable((String)"message.instantworldmirror.force_returned"), false);
                    ++playersReturned;
                }
            }
            if ((sessionIdOpt = DimensionPool.getDimensionSession(dimIndex)).isPresent()) {
                UUID sessionId = sessionIdOpt.get();
                MirrorSession session = activeSessions.get(sessionId);
                if (session != null) {
                    for (Map.Entry<UUID, UUID> entry : new HashMap<UUID, UUID>(playerOwnedSession).entrySet()) {
                        if (!entry.getValue().equals(sessionId)) continue;
                        playerOwnedSession.remove(entry.getKey());
                    }
                    MirrorWorldManager.destroySession(session, server);
                }
            } else {
                DimensionPool.DimensionState state = DimensionPool.getDimensionState(dimIndex);
                InstantWorldMirror.LOGGER.info("Force clearing dimension {} (state: {}) without active session", (Object)dimIndex, (Object)state);
                DimensionPool.markDimensionCleaning(dimIndex);
                if (mirrorWorld != null) {
                    WorldCopyService.cleanupMirrorWorld(mirrorWorld, dimIndex);
                }
            }
            InstantWorldMirror.LOGGER.info("Force cleared dimension {} - returned {} players", (Object)dimIndex, (Object)playersReturned);
            int n = playersReturned;
            return n;
        }
        finally {
            sessionLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void cleanupStaleSessions(MinecraftServer server) {
        sessionLock.writeLock().lock();
        try {
            MirrorSession session;
            int cleanedSessions = 0;
            int cleanedDimensions = 0;
            ArrayList<UUID> emptySessionIds = new ArrayList<UUID>();
            for (Map.Entry<UUID, MirrorSession> entry : activeSessions.entrySet()) {
                UUID creatorId;
                session = entry.getValue();
                if (!session.isEmpty() || session.isDestroyed() || playerOwnedSession.containsKey(creatorId = session.getCreatorId())) continue;
                emptySessionIds.add(entry.getKey());
            }
            for (UUID sessionId : emptySessionIds) {
                session = activeSessions.get(sessionId);
                if (session == null) continue;
                InstantWorldMirror.LOGGER.warn("Cleaning up stale empty session: {}", (Object)sessionId);
                MirrorWorldManager.destroySession(session, server);
                ++cleanedSessions;
            }
            int poolSize = ModDimensions.getPoolSize();
            for (int dimIndex = 0; dimIndex < poolSize; ++dimIndex) {
                ServerLevel mirrorLevel;
                UUID sessionId;
                DimensionPool.DimensionState state = DimensionPool.getDimensionState(dimIndex);
                if (state != DimensionPool.DimensionState.IN_USE || (sessionId = DimensionPool.getSessionForDimension(dimIndex)) != null && activeSessions.containsKey(sessionId) || (mirrorLevel = DimensionPool.getDimensionLevel(server, dimIndex)) == null || !mirrorLevel.players().isEmpty()) continue;
                InstantWorldMirror.LOGGER.warn("Cleaning up orphaned dimension {} (no session or players)", (Object)dimIndex);
                WorldCopyService.cancelCopyTask(dimIndex);
                DimensionPool.forceReleaseDimension(dimIndex);
                WorldCopyService.cleanupMirrorWorld(mirrorLevel, dimIndex);
                ++cleanedDimensions;
            }
            if (cleanedSessions > 0 || cleanedDimensions > 0) {
                InstantWorldMirror.LOGGER.info("Stale cleanup completed: {} sessions, {} dimensions", (Object)cleanedSessions, (Object)cleanedDimensions);
            }
        }
        finally {
            sessionLock.writeLock().unlock();
        }
    }

    public static void clearAllSessions() {
        sessionLock.writeLock().lock();
        try {
            for (MirrorSession session : activeSessions.values()) {
                session.markDestroyed();
            }
            activeSessions.clear();
            portalToSession.clear();
            playerToSession.clear();
            playerOwnedSession.clear();
            playerOriginalPositions.clear();
            playerOriginalDimensions.clear();
            playerItemTransferPermission.clear();
            DimensionPool.clearAll();
            InstantWorldMirror.LOGGER.info("All sessions and dimension pool cleared");
        }
        finally {
            sessionLock.writeLock().unlock();
        }
    }
}

