#version 150



#define COLOR_EXPONENT 0.9 //This makes colors brighter and more saturated. "1.0" would be default

#define SIZE 40.0
#define THRESHOLD 0.8 //Anything brighter than this creates bloom
#define EXPONENT 2.5 //The power raised on the brightness above the threshold
#define INTENSITY 25.0 //How bright the bloom is

#define TAPS 16 //How many different angles
#define DISTANCE 8 //How many samples along each angle



uniform sampler2D DiffuseSampler;

varying vec2 texCoord;
varying vec2 oneTexel;

float luminance( vec3 rgb ) {
    return dot( rgb, vec3( 0.2125, 0.7154, 0.0721 ) );
}

vec3 rgb2hsv( vec3 c ) {
    vec4 K = vec4( 0.0, -1.0 / 3.0, 2.0 / 3.0, -1.0 );
    vec4 p = mix( vec4( c.bg, K.wz ), vec4( c.gb, K.xy ), step( c.b, c.g ) );
    vec4 q = mix( vec4( p.xyw, c.r ), vec4( c.r, p.yzx ), step( p.x, c.r ) );

    float d = q.x - min( q.w, q.y );
    float e = 1.0e-10;
    return vec3( abs( q.z + ( q.w - q.y ) / ( 6.0 * d + e ) ), d / ( q.x + e ), q.x );
}

vec3 hsv2rgb( vec3 c ) {
    vec4 K = vec4( 1.0, 2.0 / 3.0, 1.0 / 3.0, 3.0 );
    vec3 p = abs( fract( c.xxx + K.xyz ) * 6.0 - K.www );
    return c.z * mix( K.xxx, clamp( p - K.xxx, 0.0, 1.0 ), c.y );
}

const float angle = radians( 360.0 / float( TAPS ) );
const float angleSin = sin( angle );
const float angleCos = cos( angle );
const mat2 rotationMatrix = mat2( angleCos, angleSin, -angleSin, angleCos );

const float invThres = 1.0 / (1.0 - THRESHOLD);

vec3 bloom() {
    vec2 tapOffset = vec2( 0.0, SIZE / float(DISTANCE));
    vec4 color = vec4( 0.0 );
    for ( int ii = 0; ii < TAPS; ++ii ) {
        for ( int jj = 0; jj < DISTANCE; ++jj ) {
            vec4 col = texture2D( DiffuseSampler, texCoord + oneTexel * ( tapOffset * float( jj + 1 ) ) );
            float mult = pow(max(0.0, luminance(col.rgb) - THRESHOLD) * invThres, EXPONENT) * INTENSITY;
            color += col * mult;
        }
        tapOffset = rotationMatrix * tapOffset;
    }
    color /= float( TAPS * DISTANCE );
    return color.rgb;
}

void main() {
    vec3 color = texture2D( DiffuseSampler, texCoord ).rgb;

    vec3 hsv = rgb2hsv( color );
    hsv.y = pow( hsv.y, COLOR_EXPONENT );
    hsv.z =  pow( hsv.z, COLOR_EXPONENT );
    color = hsv2rgb( hsv );

    gl_FragColor = vec4( color + bloom(), 1.0 );
}