/*
 * Decompiled with CFR 0.152.
 */
package net.oxcodsnet.roadarchitect.util;

import java.util.Map;
import java.util.OptionalInt;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.DoubleSupplier;
import java.util.function.IntSupplier;
import java.util.function.Supplier;
import net.minecraft.core.BlockPos;
import net.minecraft.core.Holder;
import net.minecraft.core.QuartPos;
import net.minecraft.resources.ResourceKey;
import net.minecraft.server.MinecraftServer;
import net.minecraft.server.level.ServerLevel;
import net.minecraft.world.level.ChunkPos;
import net.minecraft.world.level.Level;
import net.minecraft.world.level.LevelHeightAccessor;
import net.minecraft.world.level.biome.Biome;
import net.minecraft.world.level.biome.BiomeSource;
import net.minecraft.world.level.biome.Climate;
import net.minecraft.world.level.chunk.ChunkAccess;
import net.minecraft.world.level.chunk.ChunkGenerator;
import net.minecraft.world.level.levelgen.Heightmap;
import net.minecraft.world.level.levelgen.RandomState;
import net.oxcodsnet.roadarchitect.storage.CacheStorage;
import net.oxcodsnet.roadarchitect.util.AsyncExecutor;
import net.oxcodsnet.roadarchitect.util.DebugLog;
import net.oxcodsnet.roadarchitect.util.cache.ChunkHeightGenerator;
import net.oxcodsnet.roadarchitect.util.cache.ChunkHeightSnapshot;
import net.oxcodsnet.roadarchitect.util.cache.WorldCacheState;
import net.oxcodsnet.roadarchitect.util.profiler.PipelineProfiler;
import net.oxcodsnet.roadarchitect.worldgen.RoadFeature;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CacheManager {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)("roadarchitect/" + CacheManager.class.getSimpleName()));
    private static final Map<ResourceKey<Level>, WorldCacheState> STATES = new ConcurrentHashMap<ResourceKey<Level>, WorldCacheState>();
    private static final int CHUNK_SIDE = 16;
    private static final int COLUMNS_PER_CHUNK = 256;

    private CacheManager() {
    }

    public static void onWorldLoad(ServerLevel world) {
        if (world.isClientSide()) {
            return;
        }
        CacheManager.load(world);
    }

    public static void onWorldUnload(ServerLevel world) {
        if (world.isClientSide()) {
            return;
        }
        CacheManager.save(world);
    }

    public static void onServerStopping(MinecraftServer server) {
        for (ServerLevel world : server.getAllLevels()) {
            CacheManager.save(world);
        }
    }

    private static WorldCacheState state(ServerLevel world) {
        return STATES.computeIfAbsent((ResourceKey<Level>)world.dimension(), k -> new WorldCacheState(world, CacheStorage.get(world)));
    }

    private static void load(ServerLevel world) {
        CacheStorage storage = CacheStorage.get(world);
        STATES.put((ResourceKey<Level>)world.dimension(), new WorldCacheState(world, storage));
        DebugLog.info(LOGGER, "Cache loaded for world {}", world.dimension().location());
    }

    private static void save(ServerLevel world) {
        WorldCacheState state = STATES.remove(world.dimension());
        if (state != null) {
            state.chunkHeights().clear();
            state.storage().setDirty();
            DebugLog.info(LOGGER, "Cache saved for world {}", world.dimension().location());
        }
    }

    public static void prefill(ServerLevel world, int minX, int minZ, int maxX, int maxZ) {
        int step = 4;
        WorldCacheState state = CacheManager.state(world);
        CacheStorage storage = state.storage();
        AsyncExecutor.execute(() -> {
            ChunkGenerator gen = world.getChunkSource().getGenerator();
            RandomState cfg = world.getChunkSource().randomState();
            Climate.Sampler sampler = cfg.sampler();
            BiomeSource bsrc = gen.getBiomeSource();
            for (int x = minX; x <= maxX; x += step) {
                for (int z = minZ; z <= maxZ; z += step) {
                    long key = CacheManager.hash(x, z);
                    int finalX = x;
                    int finalZ = z;
                    AsyncExecutor.execute(() -> {
                        int h = gen.getBaseHeight(finalX, finalZ, Heightmap.Types.WORLD_SURFACE_WG, (LevelHeightAccessor)world, cfg);
                        Holder biome = bsrc.getNoiseBiome(QuartPos.fromBlock((int)finalX), 316, QuartPos.fromBlock((int)finalZ), sampler);
                        storage.heights().put(key, h);
                        storage.biomes().put(key, (Holder<Biome>)biome);
                    });
                }
            }
            DebugLog.info(LOGGER, "Prefill complete [{}..{}]\u00d7[{}..{}]", minX, maxX, minZ, maxZ);
        });
    }

    public static int getHeight(ServerLevel world, long key, IntSupplier loader) {
        PipelineProfiler.increment("cache.height.requests");
        WorldCacheState state = CacheManager.state(world);
        CacheStorage storage = state.storage();
        Integer cached = (Integer)storage.heights().get(key);
        if (cached != null) {
            PipelineProfiler.increment("cache.height.hits");
            return cached;
        }
        Integer chunkCached = state.lookupHeight(key, 16);
        if (chunkCached != null) {
            storage.heights().put(key, chunkCached);
            PipelineProfiler.increment("cache.height.hits");
            return chunkCached;
        }
        ChunkHeightSnapshot snapshot = CacheManager.ensureChunkSnapshot(world, state, storage, key);
        if (snapshot != null) {
            int x = (int)(key >> 32);
            int z = (int)key;
            int localX = x & 0xF;
            int localZ = z & 0xF;
            int value = snapshot.get(localX, localZ);
            storage.heights().putIfAbsent(key, value);
            PipelineProfiler.increment("cache.height.hits");
            return value;
        }
        return storage.heights().computeIfAbsent(key, k -> {
            PipelineProfiler.increment("cache.height.loads");
            try (PipelineProfiler.Section section = PipelineProfiler.openSection("cache.height.load_time");){
                int value = loader.getAsInt();
                PipelineProfiler.recordValue("cache.height.loaded_value", value);
                Integer n = value;
                return n;
            }
        });
    }

    public static int getHeight(ServerLevel world, int x, int z) {
        long key = CacheManager.hash(x, z);
        OptionalInt prepared = RoadFeature.lookupPreparedSurface(x, z);
        if (prepared.isPresent()) {
            int value = prepared.getAsInt();
            WorldCacheState state = CacheManager.state(world);
            state.storage().heights().putIfAbsent(key, value);
            return value;
        }
        return CacheManager.getHeight(world, key, () -> {
            ChunkGenerator gen = world.getChunkSource().getGenerator();
            RandomState cfg = world.getChunkSource().randomState();
            return gen.getBaseHeight(x, z, Heightmap.Types.WORLD_SURFACE_WG, (LevelHeightAccessor)world, cfg);
        });
    }

    public static double getStability(ServerLevel world, long key, DoubleSupplier loader) {
        PipelineProfiler.increment("cache.stability.requests");
        WorldCacheState state = CacheManager.state(world);
        CacheStorage storage = state.storage();
        Double cached = (Double)storage.stabilities().get(key);
        if (cached != null) {
            PipelineProfiler.increment("cache.stability.hits");
            return cached;
        }
        return storage.stabilities().computeIfAbsent(key, k -> {
            PipelineProfiler.increment("cache.stability.loads");
            try (PipelineProfiler.Section section = PipelineProfiler.openSection("cache.stability.load_time");){
                double value = loader.getAsDouble();
                PipelineProfiler.recordValue("cache.stability.loaded_value", value);
                Double d = value;
                return d;
            }
        });
    }

    public static Holder<Biome> getBiome(ServerLevel world, long key, Supplier<Holder<Biome>> loader) {
        PipelineProfiler.increment("cache.biome.requests");
        WorldCacheState state = CacheManager.state(world);
        CacheStorage storage = state.storage();
        Holder cached = (Holder)storage.biomes().get(key);
        if (cached != null) {
            PipelineProfiler.increment("cache.biome.hits");
            return cached;
        }
        return storage.biomes().computeIfAbsent(key, k -> {
            PipelineProfiler.increment("cache.biome.loads");
            try (PipelineProfiler.Section section = PipelineProfiler.openSection("cache.biome.load_time");){
                Holder value;
                Holder holder = value = (Holder)loader.get();
                return holder;
            }
        });
    }

    public static long hash(int x, int z) {
        return (long)x << 32 | (long)z & 0xFFFFFFFFL;
    }

    public static BlockPos keyToPos(long k) {
        return new BlockPos((int)(k >> 32), 0, (int)k);
    }

    public static void onChunkLoad(ServerLevel world, ChunkAccess chunk) {
        if (world.isClientSide()) {
            return;
        }
        WorldCacheState state = STATES.get(world.dimension());
        if (state == null) {
            return;
        }
        ChunkPos pos = chunk.getPos();
        int startX = pos.getMinBlockX();
        int startZ = pos.getMinBlockZ();
        int minY = state.minWorldY();
        Heightmap wg = chunk.getOrCreateHeightmapUnprimed(Heightmap.Types.WORLD_SURFACE_WG);
        Heightmap surface = chunk.getOrCreateHeightmapUnprimed(Heightmap.Types.WORLD_SURFACE);
        if (wg == null && surface == null) {
            return;
        }
        if (wg == null) {
            wg = surface;
        }
        if (surface == null) {
            surface = wg;
        }
        int[] snapshot = new int[256];
        boolean dirty = false;
        CacheStorage storage = state.storage();
        for (int localZ = 0; localZ < 16; ++localZ) {
            for (int localX = 0; localX < 16; ++localX) {
                int idx = localZ * 16 + localX;
                int height = wg.getFirstAvailable(localX, localZ);
                if (height <= minY) {
                    height = surface.getFirstAvailable(localX, localZ);
                }
                snapshot[idx] = height;
                long key = CacheManager.hash(startX + localX, startZ + localZ);
                Integer previous = storage.heights().put(key, height);
                if (previous != null && previous == height) continue;
                dirty = true;
            }
        }
        state.putChunkSnapshot(pos.toLong(), new ChunkHeightSnapshot(snapshot, 16));
        if (dirty) {
            storage.setDirty();
        }
    }

    public static void onChunkUnload(ServerLevel world, ChunkPos pos) {
        if (world.isClientSide()) {
            return;
        }
        WorldCacheState state = STATES.get(world.dimension());
        if (state != null) {
            state.removeChunkSnapshot(pos.toLong());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static ChunkHeightSnapshot ensureChunkSnapshot(ServerLevel world, WorldCacheState state, CacheStorage storage, long key) {
        int x = (int)(key >> 32);
        int z = (int)key;
        ChunkPos chunkPos = new ChunkPos(x >> 4, z >> 4);
        long chunkKey = chunkPos.toLong();
        ChunkHeightSnapshot cached = state.chunkHeights().get(chunkKey);
        if (cached != null) {
            return cached;
        }
        CompletableFuture<ChunkHeightSnapshot> future = new CompletableFuture<ChunkHeightSnapshot>();
        CompletableFuture existing = state.chunkComputations().putIfAbsent(chunkKey, future);
        if (existing != null) {
            try {
                return (ChunkHeightSnapshot)existing.join();
            }
            catch (RuntimeException e) {
                DebugLog.info(LOGGER, "Height snapshot future failed for chunk {}", chunkPos, e);
                return null;
            }
        }
        try {
            ChunkHeightSnapshot generated = ChunkHeightGenerator.generate(world, state, storage, chunkPos, 16, 256);
            if (generated != null) {
                state.putChunkSnapshot(chunkKey, generated);
            }
            future.complete(generated);
            ChunkHeightSnapshot chunkHeightSnapshot = generated;
            return chunkHeightSnapshot;
        }
        catch (RuntimeException e) {
            future.completeExceptionally(e);
            LOGGER.error("Failed to compute height snapshot for chunk {}", (Object)chunkPos, (Object)e);
            ChunkHeightSnapshot chunkHeightSnapshot = null;
            return chunkHeightSnapshot;
        }
        finally {
            state.chunkComputations().remove(chunkKey);
        }
    }
}

