/*
 * Decompiled with CFR 0.152.
 */
package net.oxcodsnet.roadarchitect.util.profiler;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.StringJoiner;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.concurrent.atomic.DoubleAccumulator;
import java.util.concurrent.atomic.DoubleAdder;
import java.util.concurrent.atomic.LongAccumulator;
import java.util.concurrent.atomic.LongAdder;
import java.util.stream.Collectors;
import net.minecraft.class_2338;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class PipelineProfiler
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger((String)("roadarchitect/" + PipelineProfiler.class.getSimpleName()));
    private static final ThreadFactory SAMPLER_THREAD_FACTORY = runnable -> {
        Thread thread = new Thread(runnable, "roadarchitect-pipeline-profiler");
        thread.setDaemon(true);
        thread.setPriority(Math.max(2, 4));
        return thread;
    };
    private static final ScheduledExecutorService SAMPLER = Executors.newSingleThreadScheduledExecutor(SAMPLER_THREAD_FACTORY);
    private static final long SAMPLE_PERIOD_MILLIS = 1000L;
    private static final AtomicReference<PipelineProfiler> ACTIVE = new AtomicReference();
    private final String trigger;
    private final String worldId;
    private final class_2338 origin;
    private final long startedNanos;
    private final Instant startedWallClock;
    private final ConcurrentMap<String, TimingStat> timings = new ConcurrentHashMap<String, TimingStat>();
    private final ConcurrentMap<String, LongAdder> counters = new ConcurrentHashMap<String, LongAdder>();
    private final ConcurrentMap<String, ValueStat> values = new ConcurrentHashMap<String, ValueStat>();
    private final Object sampleLock = new Object();
    private final Map<String, Long> lastCounterSnapshot = new HashMap<String, Long>();
    private final Map<String, Long> lastTimingSnapshot = new HashMap<String, Long>();
    private final Map<String, Long> lastValueSnapshot = new HashMap<String, Long>();
    private volatile long lastSampleNanos;
    private volatile ScheduledFuture<?> samplerFuture;

    private PipelineProfiler(String trigger, String worldId, class_2338 origin) {
        this.trigger = trigger;
        this.worldId = worldId;
        this.origin = origin == null ? null : origin.method_10062();
        this.startedNanos = System.nanoTime();
        this.startedWallClock = Instant.now();
        this.lastSampleNanos = 0L;
        this.samplerFuture = this.scheduleSampler();
    }

    public static PipelineProfiler start(String trigger, String worldId, class_2338 origin) {
        PipelineProfiler profiler = new PipelineProfiler(trigger, worldId, origin);
        PipelineProfiler previous = ACTIVE.getAndSet(profiler);
        if (previous != null) {
            LOGGER.warn("Previous profiler session {} -> {} leaked, overriding", (Object)previous.trigger, (Object)trigger);
            previous.cancelSampler();
        }
        return profiler;
    }

    public static Optional<PipelineProfiler> current() {
        return Optional.ofNullable(ACTIVE.get());
    }

    public static Section openSection(String name) {
        PipelineProfiler profiler = ACTIVE.get();
        return profiler == null ? Section.NOOP : profiler.section(name);
    }

    public static void increment(String name) {
        PipelineProfiler.increment(name, 1L);
    }

    public static void increment(String name, long delta) {
        PipelineProfiler profiler = ACTIVE.get();
        if (profiler == null) {
            return;
        }
        profiler.addCounter(name, delta);
    }

    public static void recordValue(String name, double value) {
        PipelineProfiler profiler = ACTIVE.get();
        if (profiler == null) {
            return;
        }
        profiler.addValue(name, value);
    }

    public static void recordDuration(String name, long nanos) {
        PipelineProfiler profiler = ACTIVE.get();
        if (profiler == null) {
            return;
        }
        profiler.addTiming(name, nanos);
    }

    public Section section(String name) {
        return new Section(this, name);
    }

    public void addCounter(String name, long delta) {
        this.counters.computeIfAbsent(name, key -> new LongAdder()).add(delta);
    }

    public void addValue(String name, double value) {
        this.values.computeIfAbsent(name, key -> new ValueStat()).add(value);
    }

    public void addTiming(String name, long nanos) {
        this.timings.computeIfAbsent(name, key -> new TimingStat()).add(nanos);
    }

    public ProfilerReport snapshot() {
        long now = System.nanoTime();
        return this.snapshot(now - this.startedNanos);
    }

    private ProfilerReport snapshot(long totalNanos) {
        List timingEntries = this.timings.entrySet().stream().map(e -> new TimingEntry((String)e.getKey(), ((TimingStat)e.getValue()).count.longValue(), ((TimingStat)e.getValue()).total.longValue(), ((TimingStat)e.getValue()).max.longValue())).sorted(Comparator.comparingLong(TimingEntry::totalNanos).reversed()).collect(Collectors.toCollection(ArrayList::new));
        List counterEntries = this.counters.entrySet().stream().map(e -> new CounterEntry((String)e.getKey(), ((LongAdder)e.getValue()).longValue())).sorted(Comparator.comparing(CounterEntry::name)).collect(Collectors.toCollection(ArrayList::new));
        List valueEntries = this.values.entrySet().stream().map(e -> new ValueEntry((String)e.getKey(), ((ValueStat)e.getValue()).count.longValue(), ((ValueStat)e.getValue()).sum.doubleValue(), ((ValueStat)e.getValue()).min.get(), ((ValueStat)e.getValue()).max.get())).sorted(Comparator.comparing(ValueEntry::name)).collect(Collectors.toCollection(ArrayList::new));
        return new ProfilerReport(this.trigger, this.worldId, this.origin, this.startedWallClock, totalNanos, Collections.unmodifiableList(timingEntries), Collections.unmodifiableList(counterEntries), Collections.unmodifiableList(valueEntries));
    }

    @Override
    public void close() {
        long totalNanos = System.nanoTime() - this.startedNanos;
        this.cancelSampler();
        ACTIVE.compareAndSet(this, null);
        ProfilerReport report = this.snapshot(totalNanos);
        this.logPeriodicSample(report, totalNanos, true);
        this.logReport(report);
    }

    private ScheduledFuture<?> scheduleSampler() {
        return SAMPLER.scheduleAtFixedRate(() -> {
            try {
                this.logPeriodicSample(false);
            }
            catch (Exception e) {
                LOGGER.error("Failed to record pipeline profiler sample", (Throwable)e);
            }
        }, 1000L, 1000L, TimeUnit.MILLISECONDS);
    }

    private void cancelSampler() {
        ScheduledFuture<?> future = this.samplerFuture;
        if (future != null) {
            future.cancel(false);
            this.samplerFuture = null;
        }
    }

    private void logPeriodicSample(boolean finalSample) {
        long totalNanos = System.nanoTime() - this.startedNanos;
        ProfilerReport report = this.snapshot(totalNanos);
        this.logPeriodicSample(report, totalNanos, finalSample);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void logPeriodicSample(ProfilerReport report, long totalNanos, boolean finalSample) {
        String message = null;
        Object object = this.sampleLock;
        synchronized (object) {
            if (totalNanos <= this.lastSampleNanos) {
                this.lastSampleNanos = Math.max(this.lastSampleNanos, totalNanos);
                return;
            }
            long windowNanos = totalNanos - this.lastSampleNanos;
            Map<String, Long> counterDelta = this.computeCounterDelta(report.counters());
            Map<String, Long> timingDelta = this.computeTimingDelta(report.timings());
            Map<String, Long> valueDelta = this.computeValueDelta(report.values());
            this.lastSampleNanos = totalNanos;
            if (!LOGGER.isInfoEnabled()) {
                return;
            }
            if (counterDelta.isEmpty() && timingDelta.isEmpty() && valueDelta.isEmpty()) {
                return;
            }
            StringBuilder sb = new StringBuilder();
            sb.append("Pipeline profiler window [trigger=").append(report.trigger()).append(", world=").append(report.worldId());
            if (report.origin() != null) {
                sb.append(", origin=").append(report.origin());
            }
            sb.append(", window=").append(PipelineProfiler.formatMillis(windowNanos)).append(" ms").append(", total=").append(PipelineProfiler.formatMillis(totalNanos)).append(" ms");
            if (finalSample) {
                sb.append(", final=true");
            }
            sb.append(']');
            if (!counterDelta.isEmpty()) {
                sb.append(" counters{").append(PipelineProfiler.formatDelta(counterDelta)).append('}');
            }
            if (!timingDelta.isEmpty()) {
                sb.append(" timings{").append(PipelineProfiler.formatDelta(timingDelta)).append('}');
            }
            if (!valueDelta.isEmpty()) {
                sb.append(" values{").append(PipelineProfiler.formatDelta(valueDelta)).append('}');
            }
            message = sb.toString();
        }
        if (message != null) {
            LOGGER.info(message);
        }
    }

    private Map<String, Long> computeCounterDelta(List<CounterEntry> countersSnapshot) {
        LinkedHashMap<String, Long> delta = new LinkedHashMap<String, Long>();
        HashMap<String, Long> current = new HashMap<String, Long>();
        for (CounterEntry entry : countersSnapshot) {
            long previousValue;
            long currentValue = entry.value();
            long diff = currentValue - (previousValue = this.lastCounterSnapshot.getOrDefault(entry.name(), 0L).longValue());
            if (diff != 0L) {
                delta.put(entry.name(), diff);
            }
            current.put(entry.name(), currentValue);
        }
        this.lastCounterSnapshot.clear();
        this.lastCounterSnapshot.putAll(current);
        return delta;
    }

    private Map<String, Long> computeTimingDelta(List<TimingEntry> timingSnapshot) {
        LinkedHashMap<String, Long> delta = new LinkedHashMap<String, Long>();
        HashMap<String, Long> current = new HashMap<String, Long>();
        for (TimingEntry entry : timingSnapshot) {
            long previousValue;
            long currentValue = entry.count();
            long diff = currentValue - (previousValue = this.lastTimingSnapshot.getOrDefault(entry.name(), 0L).longValue());
            if (diff != 0L) {
                delta.put(entry.name(), diff);
            }
            current.put(entry.name(), currentValue);
        }
        this.lastTimingSnapshot.clear();
        this.lastTimingSnapshot.putAll(current);
        return delta;
    }

    private Map<String, Long> computeValueDelta(List<ValueEntry> valueSnapshot) {
        LinkedHashMap<String, Long> delta = new LinkedHashMap<String, Long>();
        HashMap<String, Long> current = new HashMap<String, Long>();
        for (ValueEntry entry : valueSnapshot) {
            long previousValue;
            long currentValue = entry.count();
            long diff = currentValue - (previousValue = this.lastValueSnapshot.getOrDefault(entry.name(), 0L).longValue());
            if (diff != 0L) {
                delta.put(entry.name(), diff);
            }
            current.put(entry.name(), currentValue);
        }
        this.lastValueSnapshot.clear();
        this.lastValueSnapshot.putAll(current);
        return delta;
    }

    private static String formatDelta(Map<String, Long> delta) {
        StringJoiner joiner = new StringJoiner(", ");
        delta.forEach((name, value) -> joiner.add(name + "=" + value));
        return joiner.toString();
    }

    private void logReport(ProfilerReport report) {
        if (!LOGGER.isInfoEnabled()) {
            return;
        }
        StringBuilder sb = new StringBuilder();
        sb.append("Pipeline profiler summary [trigger=").append(report.trigger()).append(", world=").append(report.worldId());
        if (report.origin() != null) {
            sb.append(", origin=").append(report.origin());
        }
        sb.append(", wallClock=").append(report.startedWallClock()).append("]\n");
        sb.append("  Total: ").append(PipelineProfiler.formatMillis(report.totalNanos())).append(" ms\n");
        if (!report.timings().isEmpty()) {
            sb.append("  Timings:\n");
            for (TimingEntry timing : report.timings()) {
                double avgMs = timing.count() == 0L ? 0.0 : (double)timing.totalNanos() / (double)timing.count() / 1000000.0;
                sb.append("    - ").append(timing.name()).append(": total=").append(PipelineProfiler.formatMillis(timing.totalNanos())).append(" ms, avg=").append(String.format(Locale.ROOT, "%.3f", avgMs)).append(" ms, max=").append(PipelineProfiler.formatMillis(timing.maxNanos())).append(" ms (samples=").append(timing.count()).append(")\n");
            }
        }
        if (!report.counters().isEmpty()) {
            sb.append("  Counters:\n");
            for (CounterEntry counter : report.counters()) {
                sb.append("    - ").append(counter.name()).append(": ").append(counter.value()).append('\n');
            }
        }
        if (!report.values().isEmpty()) {
            sb.append("  Values:\n");
            for (ValueEntry value : report.values()) {
                double avg = value.count() == 0L ? 0.0 : value.sum() / (double)value.count();
                sb.append("    - ").append(value.name()).append(": avg=").append(String.format(Locale.ROOT, "%.3f", avg)).append(", min=").append(PipelineProfiler.formatDouble(value.min())).append(", max=").append(PipelineProfiler.formatDouble(value.max())).append(", samples=").append(value.count()).append('\n');
            }
        }
        LOGGER.info(sb.toString());
    }

    private static String formatMillis(long nanos) {
        double ms = (double)nanos / 1000000.0;
        return String.format(Locale.ROOT, "%.3f", ms);
    }

    private static String formatDouble(double value) {
        if (Double.isInfinite(value) || Double.isNaN(value)) {
            return "n/a";
        }
        return String.format(Locale.ROOT, "%.3f", value);
    }

    public static final class Section
    implements AutoCloseable {
        private static final Section NOOP = new Section();
        private final PipelineProfiler owner;
        private final String name;
        private final long started;

        private Section() {
            this.owner = null;
            this.name = "noop";
            this.started = 0L;
        }

        private Section(PipelineProfiler owner, String name) {
            this.owner = owner;
            this.name = name;
            this.started = System.nanoTime();
        }

        @Override
        public void close() {
            if (this.owner == null) {
                return;
            }
            long duration = System.nanoTime() - this.started;
            this.owner.addTiming(this.name, duration);
        }
    }

    private static final class ValueStat {
        private final DoubleAdder sum = new DoubleAdder();
        private final LongAdder count = new LongAdder();
        private final DoubleAccumulator min = new DoubleAccumulator(Math::min, Double.POSITIVE_INFINITY);
        private final DoubleAccumulator max = new DoubleAccumulator(Math::max, Double.NEGATIVE_INFINITY);

        private ValueStat() {
        }

        void add(double value) {
            this.sum.add(value);
            this.count.increment();
            this.min.accumulate(value);
            this.max.accumulate(value);
        }
    }

    private static final class TimingStat {
        private final LongAdder total = new LongAdder();
        private final LongAdder count = new LongAdder();
        private final LongAccumulator max = new LongAccumulator(Long::max, 0L);

        private TimingStat() {
        }

        void add(long nanos) {
            this.total.add(nanos);
            this.count.increment();
            this.max.accumulate(nanos);
        }
    }

    public record ProfilerReport(String trigger, String worldId, class_2338 origin, Instant startedWallClock, long totalNanos, List<TimingEntry> timings, List<CounterEntry> counters, List<ValueEntry> values) {
    }

    public record CounterEntry(String name, long value) {
    }

    public record TimingEntry(String name, long count, long totalNanos, long maxNanos) {
    }

    public record ValueEntry(String name, long count, double sum, double min, double max) {
    }
}

