#version 150

uniform sampler2D DiffuseSampler;
uniform float DistortionStrength;
uniform float Frequency;
uniform float Time;
uniform float Progress;
uniform vec2 OutSize;

in vec2 texCoord;
out vec4 fragColor;

void main() {
    vec2 uv = texCoord;

    vec2 distortion = vec2(0.0);

    distortion.x = sin(uv.y * Frequency + Time) * DistortionStrength;
    distortion.y = cos(uv.x * Frequency + Time) * DistortionStrength;

    vec2 centerVec = uv - vec2(0.5, 0.5);
    float centerDist = length(centerVec);
    vec2 radialDistortion = centerVec * sin(centerDist * Frequency * 2.0 + Time) * DistortionStrength * 0.5;

    vec2 shippudenSpiral = vec2(0.0);

    float spiralPower = 0.025 * DistortionStrength;
    float spiralTurns = 12.0; // 螺旋圈数
    float spiralSpeed = 3.0; // 螺旋速度

    float distToCenter = length(centerVec);
    float angle = atan(centerVec.y, centerVec.x);

    float spiralRotation = spiralTurns * distToCenter - Time * spiralSpeed;

    float cosRot = cos(spiralRotation);
    float sinRot = sin(spiralRotation);

    vec2 rotatedCoord = vec2(
    centerVec.x * cosRot - centerVec.y * sinRot,
    centerVec.x * sinRot + centerVec.y * cosRot
    );

    // 向中心收缩（神威的吸入效果）
    float contraction = 0.015 * DistortionStrength * (1.0 - exp(-distToCenter * 8.0));
    rotatedCoord *= (1.0 - contraction * Progress);

    // 计算畸变向量
    shippudenSpiral = (rotatedCoord - centerVec) * spiralPower * Progress;

    // 轻微的中心扭曲增强
    float centerWarp = 0.01 * DistortionStrength * sin(Time * 3.0) * exp(-distToCenter * 10.0);
    shippudenSpiral += centerVec * centerWarp * Progress;

    // 组合所有畸变效果
    vec2 finalDistortion = distortion + radialDistortion + shippudenSpiral;

    // 根据进度调整最终畸变
    finalDistortion *= Progress;

    // 应用畸变
    vec2 distortedUV = uv + finalDistortion;

    // 确保UV在有效范围内
    distortedUV = clamp(distortedUV, 0.0, 1.0);

    // 采样纹理
    fragColor = texture(DiffuseSampler, distortedUV);

    // 神威特有的视觉特效
    if (Progress > 0.5) {
        // 1. 中心区域的颜色偏移（神威的空间扭曲感）
        float centerEffect = exp(-distToCenter * 15.0) * Progress;
        if (centerEffect > 0.1) {
            vec2 offset = centerVec * 0.005 * centerEffect;
            fragColor.r = texture(DiffuseSampler, clamp(distortedUV + offset, 0.0, 1.0)).r;
            fragColor.b = texture(DiffuseSampler, clamp(distortedUV - offset, 0.0, 1.0)).b;
        }

        // 2. 边缘的轻微模糊/光晕效果（神威的空间撕裂感）
        if (distToCenter > 0.3 && Progress > 0.7) {
            float edgeEffect = smoothstep(0.3, 0.5, distToCenter) * (Progress - 0.7) * 3.0;
            vec2 blurOffset = normalize(centerVec) * 0.003 * edgeEffect;
            vec4 blurred = texture(DiffuseSampler, clamp(distortedUV + blurOffset, 0.0, 1.0)) * 0.5 +
            texture(DiffuseSampler, clamp(distortedUV - blurOffset, 0.0, 1.0)) * 0.5;
            fragColor = mix(fragColor, blurred, edgeEffect * 0.3);
        }
    }

    if (Progress > 0.7) {
        float chromaStrength = (Progress - 0.7) * 3.0 * DistortionStrength;
        vec2 chromaOffset = vec2(chromaStrength * 0.01, 0.0);

        fragColor.r = texture(DiffuseSampler, clamp(distortedUV + chromaOffset, 0.0, 1.0)).r;
        fragColor.b = texture(DiffuseSampler, clamp(distortedUV - chromaOffset, 0.0, 1.0)).b;
    }
}