$basePath = ".\data\chems_guns"

# Normalize to absolute paths
$lootRoot   = (Join-Path $basePath "loot_table\guns" | Resolve-Path).Path
$recipeRoot = (Join-Path $basePath "recipe\guns"     | Resolve-Path).Path

#
# Extract the ENTIRE components block from the loot-table JSON.
#
function Get-ComponentsBlock {
    param($lootFile)

    $text = Get-Content $lootFile -Raw

    # Balanced-brace match for: "components": { ... }
    $regex = '"components"\s*:\s*(\{(?:[^{}]|(?<open>\{)|(?<-open>\}))+(?(open)(?!))\})'

    $match = [regex]::Match($text, $regex)

    if ($match.Success) {
        # Return only the object part: { ... }
        return $match.Groups[1].Value
    }

    return $null
}

#
# Replace the recipe JSON's components block with the one extracted from the loot-table.
#
function Replace-ComponentsBlock {
    param(
        [string]$recipeFile,
        [string]$componentsJson
    )

    $recipeText = Get-Content $recipeFile -Raw

    # Regex for recipe's components object
    $componentsRegex = '("components"\s*:\s*)(\{(?:[^{}]|(?<open>\{)|(?<-open>\}))+(?(open)(?!))\})'

    if ([regex]::IsMatch($recipeText, $componentsRegex)) {
        # Replace existing block
        $newText = [regex]::Replace(
            $recipeText,
            $componentsRegex,
            "`$1$componentsJson"
        )
    }
    else {
        Write-Host "ERROR: Recipe missing components block: $recipeFile"
        return
    }

    Set-Content -Path $recipeFile -Value $newText -Encoding UTF8
}

#
# Main loop
#
Get-ChildItem -Path $lootRoot -Filter "*.json" -Recurse | ForEach-Object {
    $lootFile = $_.FullName

    $relPath = $lootFile.Substring($lootRoot.Length).TrimStart('\','/')
    $recipeFile = Join-Path $recipeRoot $relPath

    if (-not (Test-Path $recipeFile)) {
        Write-Host "Skipping: No recipe for $relPath"
        return
    }

    $components = Get-ComponentsBlock $lootFile

    if ($null -eq $components) {
        Write-Host "No components block found in loot table: $relPath"
        return
    }

    Replace-ComponentsBlock -recipeFile $recipeFile -componentsJson $components

    Write-Host "Updated: $relPath"
}

Write-Host "All matching recipes processed!"
